// Adobe.LipSync.js
/*jslint sub: true */
/*global define, console, nmlImpl */
//
define( ["lib/Zoot", "src/utils"],
  function (Z, utils) {
		"use strict";

	var mouthShapeTagDefinitions = [  // warning: indices of this array are semantic and match eventgraph data and kVisemes below
			{								// 0
				id: "Adobe.Face.NeutralMouth",	// see parallel definition in FaceTracker
				artMatches: ["neutral"],
				uiName: "$$$/animal/Behavior/Face/TagName/Neutral=Neutral",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
			{
				id: "Adobe.LipSynch.Ah",	// 1
				artMatches: ["ah"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/Ah=Ah",
			},
			{
				id: "Adobe.LipSynch.D",		// 2
				artMatches: ["d"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/D=D",
			},
			{
				id: "Adobe.LipSynch.Ee",	// 3
				artMatches: ["ee"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/Ee=Ee",
			},
			{
				id: "Adobe.LipSynch.F",		// 4
				artMatches: ["f"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/F=F",
			},
			{
				id: "Adobe.LipSynch.L",		// 5
				artMatches: ["l", "th"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/L=L",
			},
			{
				id: "Adobe.LipSynch.M",		// 6
				artMatches: ["m"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/M=M",
			},
			{
				id: "Adobe.LipSynch.Oh",	// 7
				artMatches: ["oh", "oh-uh"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/Oh=Oh",
			},
			{
				id: "Adobe.LipSynch.R",		// 8
				artMatches: ["r"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/R=R",
			},			
			{
				id: "Adobe.LipSynch.S",		// 9
				artMatches: ["k", "s"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/S=S",
			},
			{
				id: "Adobe.LipSynch.Uh",	// 10
				artMatches: ["uh"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/Uh=Uh",
			},			
			{
				id: "Adobe.LipSynch.W-Oo",	// 11
				artMatches: ["w-oo"],
				uiName: "$$$/animal/timeline/visemes/keyframe/names/W-Oo=W-Oo",
			},
		],
		
		mouthParentLayerTagDefinition = [
			{
				id: "Adobe.Face.MouthsParent",
				artMatches: ["mouth"],
				uiName: "$$$/animal/Behavior/Face/TagName/MouthGroup=Mouth Group",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]				

			},
		],
		
		kVisemes = {		// must match mouthShapeTagDefinitions; could be factored but requires
			Silence: 	0,	//	getting a bit fancy with the tag defintion and coordination with FaceTracker
			Ah: 		1,	//	for the neutral tag
			D: 			2,
			Ee: 		3,
			F: 			4,
			L: 			5,
			M: 			6,
			Oh: 		7,
			R: 			8,
			S: 			9,
			Uh: 		10,
			Woo: 		11
		}, kMouthIndexToViseme = {};
	
		for (var k in kVisemes) {	// build reverse map
			if (kVisemes.hasOwnProperty(k)) {
				kMouthIndexToViseme[kVisemes[k]] = k;
			}
		}

		// set all the common values
		mouthShapeTagDefinitions.forEach(function (tagDefn) {
			tagDefn.tagType = "layertag";
			if (!tagDefn.uiGroups) {
				tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Mouth", subSort:0}];
			}
		});
	
		// sequences of fall back mouth shape indices
		// whichever mouth shape index the facetracker returns, 
		// chooseMouthReplacements will use the first valid mouth shape in the corresponding sequence
		//	this is a sparse map from kViseme key name to array of mouthIndexes (kViseme values)
		var mouthShapeCascades = {
			// no fallbacks for Silence, F, M
			Ah: 	[ kVisemes.Uh, kVisemes.Oh ],
			D: 		[ kVisemes.S ],
			Ee: 	[ kVisemes.Uh ],
			L: 		[ kVisemes.D, kVisemes.S ],
			Oh: 	[ kVisemes.R ],
			R: 		[ kVisemes.Ee ],				// ?
			S: 		[ kVisemes.D ],
			Uh: 	[ kVisemes.R, kVisemes.Ee ],	// ?
			Woo: 	[ kVisemes.R ]
		},

		// mapping from keyboard inputs to mouth shape indices
		keyMap = {
			"A" : kVisemes.Ah,
			"D" : kVisemes.D,
			"E" : kVisemes.Ee,
			"F" : kVisemes.F,
			"L" : kVisemes.L,
			"M" : kVisemes.M,
			"O" : kVisemes.Oh,
			"R" : kVisemes.R,
			"S" : kVisemes.S,
			"U" : kVisemes.Uh,
			"W" : kVisemes.Woo,
			"0" : kVisemes.Silence, // force neutral expression (cannot be overriden by face tracked mouth shape)
		};


	// TODO: duplicated code from Adobe.KeyReplacer.js
	function isKeyDown(args, inKeyName) {
		return args.getParamEventValue("keyboardInput", Z.keyCodes.getKeyGraphId(Z.keyCodes.getKeyCode(inKeyName))) || 0;		
	}

	// returns viseme (could be silent) or undefined if low-priority silent should be set
	function chooseMouthShape (args) {
		var returnVal /* may remain undefined */, keyName, keyDown, inputEnabled0;

		// NOTE: right now, KeyReplacer behavior will trump keyboard input here.
		// If we want keyboard input here to override, we'll need to change the priority
		// of keyboard-driven viseme replacement requests.

		// give first priority to keyboard input
		for (keyName in keyMap) {
			if (keyMap.hasOwnProperty(keyName)) {

				keyDown = isKeyDown(args, keyName);
				if (keyDown) {
					args.setEventGraphParamRecordingValid("keyboardInput");
					returnVal = keyMap[keyName];

					// TODO: figure out better way to resolve mutual exclusion (last time of key down?)
				}
			}
		}

		inputEnabled0 = args.getParamEventValue("audioInput", "Viseme/InputEnabled");
		// if no keyboard input and audio input is enabled, try setting viseme based on audio 
		if (returnVal === undefined && (inputEnabled0 === undefined || inputEnabled0)) {
			var visemeVal, visemeDuration, faceTrackerSetMouthSilenceDurationThreshInS = args.getParam("faceTrackerSetMouthSilenceDurationThreshInMilliseconds") / 1000;

			if (!args.isParamEventLive("audioInput") || nmlImpl.broadcastModeB)
			{
				visemeDuration = args.getParamEventValue("audioInput", "Viseme/LookaheadHoldDuration");
				visemeVal = args.getParamEventValue("audioInput", "Viseme/LookaheadIndex");
			}
			else
			{
				visemeDuration = args.getParamEventValue("audioInput", "Viseme/HoldDuration");
				visemeVal = args.getParamEventValue("audioInput", "Viseme/Index");
				//console.logToUser("gotten Viseme/Index = " + visemeVal);
			}
			args.setEventGraphParamRecordingValid("audioInput");

			// if we get no viseme (visemeVal < 0) or we've detected silence for longer than faceTrackerSetMouthSilenceDurationThresh,
			// return an undefined visemeVal, which allows face tracker to control the mouth
			//	note that this logic is skipped for silence triggered via keyboard
			if ((visemeVal < 0) || 
				((visemeVal === kVisemes.Silence) && (visemeDuration < 0 || visemeDuration > faceTrackerSetMouthSilenceDurationThreshInS)))
			{
				returnVal = undefined;
			}
			else
			{
				returnVal = Math.round(visemeVal);
			}
		}

		return returnVal;
	}
	
	
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeLayerIdFromLabel (str) {
		return "L_" + makeValidIdFromLabel(str);
	}
	
	function defineMouthLayerParams () {
		var aParams = [];
		
		// WARNING: repeated in Adobe.FaceTracker.js, but unsorted

		mouthShapeTagDefinitions.forEach(function (tagDefn) {
			var matchID = tagDefn.id;
			aParams.push({id:makeLayerIdFromLabel(matchID),
						  type:"layer", uiName:tagDefn.uiName,
						  dephault:{match:"//"+matchID,  startMatchingAtParam:"mouthsParent"},
						  maxCount:1}); // only one per mouth, but there can be many mouths
		});
		
		var neut = aParams.shift();	// don't sort "Adobe.Face.NeutralMouth" in with the rest
		
		aParams.sort(function (a, b)
						{ if (a.uiName < b.uiName) return -1;
							if (a.uiName > b.uiName) return 1;
							return 0; });

		// add these to the front, after sorting the rest
		aParams.splice(0, 0,
					   {id:"mouthsParent",
							  type:"layer",
							  uiName:"$$$/animal/Behavior/LipSync/Parameter/Mouth=Mouth",
							  dephault:{match:"//Adobe.Face.MouthsParent"}
					   },
					   neut	// so Netural comes before all visemes in the param order
					  );

		return aParams;
	}

	// TODO: this needs updating; use kVisemes or remove completely
	function getTransitionMouthIndex (fromVisemeIndex, toVisemeIndex) {
		var transitionMouthIndex = null;

		// could put the following into a transition table, but since there are 
		// only a few valid transitions, seems like that might be overkill

		if ((fromVisemeIndex === 1 && toVisemeIndex === 7) || 		// K to D,
			(fromVisemeIndex === 7 && toVisemeIndex === 1)) {
			transitionMouthIndex = 2;								// add transition shape Ee
		}
		else if ((fromVisemeIndex === 2 && toVisemeIndex === 5) || 	// Ee to Oh-Uh,
				 (fromVisemeIndex === 5 && toVisemeIndex === 2) ||
				 (fromVisemeIndex === 1 && toVisemeIndex === 5) || 	// K to Oh-Uh, 
				 (fromVisemeIndex === 5 && toVisemeIndex === 1)) {
			transitionMouthIndex = 9;								// add transition shape Ah
		}
		else if ((fromVisemeIndex === 4 && toVisemeIndex === 9) || 	// W-Oo to Ah, 
				 (fromVisemeIndex === 9 && toVisemeIndex === 4)) {
			transitionMouthIndex = 5;								// add transition shape Oh-Uh
		}

		return transitionMouthIndex;
	}


	// TODO: factor with FaceTracker.js
	// returns array of maps from mouth label to matched layer
	// called during onCreate, so it uses getStaticParam
	//	outer array is one per mouth group (i.e. number of matches of mouthsParent)
	function getMouthsCandidates (args) {
		var aMouthsLayers = [],
			bHideSiblings = true,
			aMouths = args.getStaticParam("mouthsParent");
		
		aMouths.forEach(function (mouthLayer, mouthGroupIndex) {
			var mouths = { parent: mouthLayer };	// map from label -> layer for each label, also "parent" -> group
			mouthShapeTagDefinitions.forEach(function (tagDefn) {
				var aaLayers = args.getStaticParam(makeLayerIdFromLabel(tagDefn.id)),
					layer = aaLayers[mouthGroupIndex][0]; // we only support one of each shape per mouth
				
				if (layer) {
					mouths[tagDefn.id] = layer;
					layer.setTriggerable(bHideSiblings);
				}
			});
			aMouthsLayers.push(mouths);
		});
		
		return aMouthsLayers;
	}
	

	function chooseMouthReplacements (self, args) {

		// returns actual mouth node for requested mouth index, which may not exist (uses fallbacks in mouthShapeCascades)
		//  returns null if it can't find a good match or if Neutral should be shown
		function getClosestValidMouthContainerRoot(mouths, inMouthIndex) {

			var i, mouthIndex, mouthShapeLabel, layer,
				mouthName = kMouthIndexToViseme[inMouthIndex],
				mouthIndicesToTry = [inMouthIndex];
			
			if (!mouthName) {
				// commenting out for now b/c this produces noisy output in NUE Scene 1 
				// TODO: look into why this is happening at all ...
				//console.logToUser("invalid inMouthIndex: " + inMouthIndex);
			} else {
				if (mouthShapeCascades.hasOwnProperty(mouthName)) {
					mouthIndicesToTry = mouthIndicesToTry.concat(mouthShapeCascades[mouthName]);
				}

				for (i = 0; i < mouthIndicesToTry.length; i += 1) {
					mouthIndex = mouthIndicesToTry[i];

					if (mouthIndex !== kVisemes.Silence) {
						mouthShapeLabel = mouthShapeTagDefinitions[mouthIndex].id;
						if (!mouthShapeLabel) {
							// this shouldn't happen, so adding error message if it does to help diagnose
							console.logToUser("invalid mouth try-index: " + mouthIndex);
						} else {
							layer = mouths[mouthShapeLabel];

							//console.logToUser(mouthShapeLabel);

							if (layer) {
								/*if (i > 0) {
									console.logToUser("falling back to mouth "+mouthShapeLabel+" instead of "+mouthName);
								}*/
								return {"layer": layer, "index": mouthIndex};
							}
						}
					}
				}
			}

			return null;
		}

		// get ideal mouth to show; undefined means set neutral at a low priority (allow Face to override)
		// because we've detected silence for longer than faceTrackerSetMouthSilenceDurationThresh 
		// and Face is allowed to control the mouth replacements. Otherwise, lip sync has detected 
		// either a non-silent viseme or short silence, and we should not let face tracker control the mouth.
		var mouthIndexToShow = chooseMouthShape(args),
			bUseTransitionMouthShapes = args.getParam("useTransitionMouthShapes"),
			transitionMouthIndex = null;

		// show transition mouth if necessary
		if (bUseTransitionMouthShapes) {
			transitionMouthIndex = getTransitionMouthIndex(self.prevVisemeIndex, mouthIndexToShow);
			if (transitionMouthIndex) {
				mouthIndexToShow = transitionMouthIndex;
			}
		}

		self.aMouthsGroups.forEach(function (mouths) {	// once for each mouth group
			var //parentGroup = mouths["parent"], unused but nice to know its there
				neutralMouthLayer0 = mouths["Adobe.Face.NeutralMouth"], 
				validMouth, validMouthLayer0 = neutralMouthLayer0, validMouthIndex = -1, 
				priority;
			
			// get valid (found) mouth in this group that is closest to specified mouth to show
			if (mouthIndexToShow === undefined) {
				priority = 0;// < both Face & KeyReplacer, so Face can override
				// validMouthLayer0 is already set to neutral at this point
			} else {
				validMouth = getClosestValidMouthContainerRoot(mouths, mouthIndexToShow);
				if (validMouth) {
					validMouthLayer0 = validMouth.layer;
					validMouthIndex = validMouth.index;
				} 
				priority = 0.5; // > Face, < KeyReplacer
			}

			if (validMouthLayer0) {
				validMouthLayer0.trigger(priority);
			}

			self.prevVisemeIndex = validMouthIndex; // this is inside the per-group loop, so only the
													//	the last group influences the result, which seems wrong
													//	but has the benefit of missing mouths affecting the
													//	chosen transition mouths.
		});
	}


	return {
		about:			"$$$/private/animal/Behavior/LipSync/About=Lip Sync, (c) 2015.",
		description:	"$$$/animal/Behavior/LipSync/Desc=Animates mouth replacements based on speech analysis; overrides mouths set by the Face behavior",
		uiName:			"$$$/animal/Behavior/LipSyncOld/UIName=Lip Sync (obsolete)",
		defaultArmedForRecordOn: true,
		hideInBehaviorList: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			var aParams = [];
			aParams.push({ id: "audioInput", type: "eventGraph", uiName: "$$$/animal/Behavior/LipSync/Parameter/audioInput=Audio Input",
				inputKeysArray: ["Viseme/"], uiToolTip: "$$$/animal/Behavior/LipSync/Parameter/audioInput/tooltip=Analyzed viseme data from audio input", defaultArmedForRecordOn: true,
				outputKeyTraits: {
					editableB:true,
					takeData : {	// This may become an array of objects at some point, since a take can have multiple data streams.
						eventGraphKey : "Viseme/LookaheadIndex",
						type : "enum",
						items: [
							{ value: 0, uiName: "--"},
							{ value: 1, uiName: "$$$/animal/timeline/visemes/keyframe/names/Ah=Ah"},
							{ value: 2, uiName: "$$$/animal/timeline/visemes/keyframe/names/D=D"},
							{ value: 3, uiName: "$$$/animal/timeline/visemes/keyframe/names/Ee=Ee"},
							{ value: 4, uiName: "$$$/animal/timeline/visemes/keyframe/names/F=F"},
							{ value: 5, uiName: "$$$/animal/timeline/visemes/keyframe/names/L=L"},
							{ value: 6, uiName: "$$$/animal/timeline/visemes/keyframe/names/M=M"},
							{ value: 7, uiName: "$$$/animal/timeline/visemes/keyframe/names/Oh=Oh"},
							{ value: 8, uiName: "$$$/animal/timeline/visemes/keyframe/names/R=R"},
							{ value: 9, uiName: "$$$/animal/timeline/visemes/keyframe/names/S=S"},
							{ value: 10, uiName: "$$$/animal/timeline/visemes/keyframe/names/Uh=Uh"},
							{ value: 11, uiName: "$$$/animal/timeline/visemes/keyframe/names/W-Oo=W-Oo"},
						],
						valueMenuItems: [
							{ id: 1, uiMenuName: "$$$/animal/timeline/visemes/menu/items/Ah=Ah (i)" },
							{ id: 2, uiMenuName: "$$$/animal/timeline/visemes/menu/items/D=D (n, th, g)" },
							{ id: 3, uiMenuName: "$$$/animal/timeline/visemes/menu/items/Ee=Ee"},
							{ id: 4, uiMenuName: "$$$/animal/timeline/visemes/menu/items/F=F (v)" },
							{ id: 5, uiMenuName: "$$$/animal/timeline/visemes/menu/items/L=L (th)" },
							{ id: 6, uiMenuName: "$$$/animal/timeline/visemes/menu/items/M=M (b, p)" },
							{ id: 7, uiMenuName: "$$$/animal/timeline/visemes/menu/items/Oh=Oh" },
							{ id: 8, uiMenuName: "$$$/animal/timeline/visemes/menu/items/R=R" },
							{ id: 9, uiMenuName: "$$$/animal/timeline/visemes/menu/items/S=S (ch, j, sh, z)" },
							{ id: 10, uiMenuName: "$$$/animal/timeline/visemes/menu/items/Uh=Uh" },
							{ id: 11, uiMenuName: "$$$/animal/timeline/visemes/menu/items/W-Oo=W-Oo (q)" },
							{ uiMenuName: "-", bNonSythnesizeOnly:true }, // separator
							{ id: -2, uiMenuName: "$$$/animal/timeline/visemes/Silence=Silence", bNonSythnesizeOnly:true},
						],
						redundancyTraits : {v : -2, bNilKeyframeIsRedundant : true},
						uiLabel : "$$$/animal/Behavior/LipSync/Value/VisemesLabel=Visemes",
						uiName : "$$$/animal/Behavior/LipSync/Value/VisemeName=Viseme",
						uiNamePlural : "$$$/animal/Behavior/LipSync/Value/VisemeNamePlural=Visemes"
					}
				}
			});

			aParams.push({ id: "keyboardInput", type: "eventGraph", uiName: "$$$/animal/Behavior/LipSync/Parameter/keyboardInput=Keyboard Input",
				inputKeysArray: ["Keyboard/"], uiToolTip: "$$$/animal/Behavior/LipSync/Parameter/keyboardInput/tooltip=Show viseme by pressing the first letter of the viseme layer name; overrides audio analysis", defaultArmedForRecordOn: false, hidden: true
			});

			aParams.push({ id: "useTransitionMouthShapes", type: "checkbox", uiName: "$$$/animal/Behavior/LipSync/Parameter/TransitionMouths=Transition Mouths", dephault: false,
					uiToolTip: "$$$/animal/Behavior/LipSync/Parameter/TransitionMouths/tooltip=Automatically add transitional mouth shapes when switching between certain visemes", defaultArmedForRecordOn: false,
					hidden: true
			});

			// see also TimelineController.lua kShortSilenceDurationThresh
			aParams.push({ id: "faceTrackerSetMouthSilenceDurationThreshInMilliseconds", type: "slider", uiName: "$$$/animal/Behavior/LipSync/Parameter/SilenceDurationThreshold=Silence Duration Thresh", uiUnits: "ms", min: 0, max: 1000, precision: 0, dephault: 500,
				uiToolTip: "$$$/animal/Behavior/LipSync/Parameter/SilenceDurationThreshold/tooltip=Minimum duration for silence before we allow the face tracker to control the mouth", hidden: true
			});

			var aMouths = defineMouthLayerParams();
			
			return aParams.concat(aMouths);
		},
		
		defineTags: function () {
			var aAllTags = [];
			aAllTags = aAllTags.concat(mouthShapeTagDefinitions);
			aAllTags = aAllTags.concat(mouthParentLayerTagDefinition);
			return {
				aTags:aAllTags
			};
		},
		onCreateBackStageBehavior: function (self) {
			self.puppetInitTransforms = {};

			return { order: 0.5, importance : 0.0 }; // must come before FaceTracker
		},
		
		onCreateStageBehavior: function (self, args) {
			self.aMouthsGroups = getMouthsCandidates(args);
			self.prevVisemeIndex = -1;
		},
		
		onAnimate: function (self, args) {
			chooseMouthReplacements(self, args);
		}
		
	}; // end of object being returned
});
